<?php
// dashboard.php — one page: campaigns list + stats + popup create new + generator

require_once __DIR__ . "/config.php";

$campaigns_file = __DIR__ . "/campaigns.json";
if (!file_exists($campaigns_file)){
    file_put_contents($campaigns_file, json_encode([]));
}

// Load campaigns
$campaigns = json_decode(file_get_contents($campaigns_file), true);
if (!is_array($campaigns)) $campaigns = [];

// Helper to save
function save_campaigns($file, $arr){
    file_put_contents($file, json_encode($arr, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}

// Manual vertical list
$verticalOptions = ['RealEstate','Crypto','Finance','Ecommerce','Health'];

// Handle create new campaign (POST)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'create_campaign'){
    $name     = trim($_POST['name'] ?? '');
    $vertical = trim($_POST['vertical'] ?? '');
    $offer    = trim($_POST['offer'] ?? '');

    if ($name !== '' && $offer !== ''){
        // generate key slug
        $key = preg_replace('/[^a-z0-9]+/i','_', strtolower($name));
        if ($key === '') $key = 'camp_' . substr(md5(uniqid('', true)),0,6);

        // ensure unique key
        $orig = $key; $i=1;
        $keys = array_column($campaigns, 'key');
        while (in_array($key, $keys)){
            $key = $orig . '_' . $i++;
        }

        $campaigns[] = [
            'key'      => $key,
            'name'     => $name,
            'vertical' => $vertical,
            'offer'    => $offer,
            'created'  => date('Y-m-d H:i:s')
        ];
        save_campaigns($campaigns_file, $campaigns);

        header("Location: dashboard.php?camp=".$key);
        exit;
    }
}

// Selected campaign (for detail + links)
$selected_key = $_GET['camp'] ?? '';
$selected_campaign = null;
foreach ($campaigns as $c){
    if ($c['key'] === $selected_key){
        $selected_campaign = $c;
        break;
    }
}

// Load stats from CSVs
$baseCountries = __DIR__ . "/countries/";

function load_events_for_campaign($baseCountries, $campaign_key){
    $data = [];
    if ($campaign_key === '') return $data;

    $countryDirs = glob($baseCountries . "*");
    foreach ($countryDirs as $cdir){
        if (!is_dir($cdir)) continue;
        $countryCode = basename($cdir);
        $vertDirs = glob($cdir . "/*");
        foreach ($vertDirs as $vdir){
            if (!is_dir($vdir)) continue;

            foreach (['opens.csv','clicks.csv','unsub.csv'] as $fn){
                $f = $vdir . "/".$fn;
                if (!file_exists($f)) continue;
                $fp = fopen($f,"r");
                while (($r = fgetcsv($fp)) !== false){
                    if (($r[3] ?? '') === $campaign_key){
                        $data[] = $r;
                    }
                }
                fclose($fp);
            }
        }
    }
    return $data;
}

// Global stats per campaign for main table
function compute_stats_for_campaign($baseCountries, $campaign_key){
    $events = load_events_for_campaign($baseCountries, $campaign_key);
    $o=$c=$u=0;
    foreach ($events as $r){
        $a=$r[5] ?? '';
        if ($a==='open') $o++;
        elseif($a==='click') $c++;
        elseif($a==='unsub') $u++;
    }
    return [$o,$c,$u];
}

// If one selected campaign -> detail events
$detail_events = [];
if ($selected_campaign){
    $detail_events = load_events_for_campaign($baseCountries, $selected_campaign['key']);
    $opens = $clicks = $unsubs = 0;
    foreach ($detail_events as $r){
        $a = $r[5] ?? '';
        if ($a==='open') $opens++;
        elseif($a==='click') $clicks++;
        elseif($a==='unsub') $unsubs++;
    }
} else {
    $opens = $clicks = $unsubs = 0;
}

// Global totals
$total_campaigns = count($campaigns);
$global_opens = $global_clicks = $global_unsubs = 0;
foreach ($campaigns as $c){
    [$o,$cl,$u] = compute_stats_for_campaign($baseCountries, $c['key']);
    $global_opens += $o;
    $global_clicks += $cl;
    $global_unsubs += $u;
}

// Generate tracking links for selected campaign
$open_link = $click_link = $unsub_link = '';
if ($selected_campaign){
    $emailPlaceholder = '%EMAIL%';
    $emailEnc = urlencode($emailPlaceholder);
    $vEnc     = urlencode($selected_campaign['vertical']);
    $campEnc  = urlencode($selected_campaign['key']);
    $offerEnc = urlencode($selected_campaign['offer']);
    $id       = bin2hex(random_bytes(5));

    $baseUrl = "https://YOURDOMAIN/tracking"; // TODO: change

    $common = "email={$emailEnc}&v={$vEnc}&camp={$campEnc}&id={$id}";
    $open_link  = "{$baseUrl}/open.php?{$common}";
    $click_link = "{$baseUrl}/click.php?{$common}&u={$offerEnc}";
    $unsub_link = "{$baseUrl}/unsubscribe.php?{$common}";
}

?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Campaigns Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
</head>
<body class="bg-light">

<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <h2>📊 Campaigns Dashboard</h2>
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#newCampaignModal">Create New Campaign</button>
    </div>

    <!-- Top global stats -->
    <div class="row mb-4">
        <div class="col-md-3 mb-2">
            <div class="card text-center shadow-sm">
                <div class="card-body">
                    <h6 class="text-muted">Campaigns</h6>
                    <h4><?= $total_campaigns ?></h4>
                </div>
            </div>
        </div>
        <div class="col-md-3 mb-2">
            <div class="card text-center shadow-sm">
                <div class="card-body">
                    <h6 class="text-muted">Total Opens</h6>
                    <h4><?= $global_opens ?></h4>
                </div>
            </div>
        </div>
        <div class="col-md-3 mb-2">
            <div class="card text-center shadow-sm">
                <div class="card-body">
                    <h6 class="text-muted">Total Clicks</h6>
                    <h4><?= $global_clicks ?></h4>
                </div>
            </div>
        </div>
        <div class="col-md-3 mb-2">
            <div class="card text-center shadow-sm">
                <div class="card-body">
                    <h6 class="text-muted">Total Unsubs</h6>
                    <h4><?= $global_unsubs ?></h4>
                </div>
            </div>
        </div>
    </div>

    <!-- Campaigns table + detail -->
    <div class="row g-3">
        <div class="col-lg-6">
            <div class="card shadow-sm">
                <div class="card-header">All Campaigns</div>
                <div class="card-body p-0" style="max-height:400px;overflow:auto;">
                    <table class="table table-hover table-sm mb-0">
                        <thead class="table-light">
                        <tr>
                            <th>Name</th>
                            <th>Vertical</th>
                            <th>Offer</th>
                            <th>Opens</th>
                            <th>Clicks</th>
                            <th>Unsubs</th>
                            <th></th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($campaigns as $c): 
                            [$o,$cl,$u] = compute_stats_for_campaign($baseCountries, $c['key']);
                        ?>
                            <tr class="<?= $selected_campaign && $selected_campaign['key']===$c['key']?'table-primary':'' ?>">
                                <td><?= htmlspecialchars($c['name']) ?></td>
                                <td><?= htmlspecialchars($c['vertical']) ?></td>
                                <td style="max-width:140px;white-space:nowrap;overflow:hidden;text-overflow:ellipsis;">
                                    <?= htmlspecialchars($c['offer']) ?>
                                </td>
                                <td><?= $o ?></td>
                                <td><?= $cl ?></td>
                                <td><?= $u ?></td>
                                <td>
                                    <a class="btn btn-sm btn-outline-primary" href="?camp=<?= urlencode($c['key']) ?>">View</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if(!count($campaigns)): ?>
                            <tr><td colspan="7" class="text-center text-muted">No campaigns yet.</td></tr>
                        <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Detail panel -->
        <div class="col-lg-6">
            <div class="card shadow-sm mb-3">
                <div class="card-header">Campaign Detail</div>
                <div class="card-body">
                    <?php if($selected_campaign): ?>
                        <h5><?= htmlspecialchars($selected_campaign['name']) ?></h5>
                        <p class="mb-1">
                            <strong>Key:</strong> <?= htmlspecialchars($selected_campaign['key']) ?><br>
                            <strong>Vertical:</strong> <?= htmlspecialchars($selected_campaign['vertical']) ?><br>
                            <strong>Offer:</strong> <a href="<?= htmlspecialchars($selected_campaign['offer']) ?>" target="_blank">
                                <?= htmlspecialchars($selected_campaign['offer']) ?>
                            </a><br>
                            <strong>Created:</strong> <?= htmlspecialchars($selected_campaign['created']) ?>
                        </p>

                        <div class="row mb-3">
                            <div class="col-4">
                                <div class="card text-center shadow-sm">
                                    <div class="card-body p-2">
                                        <small class="text-muted">Opens</small>
                                        <h5><?= $opens ?></h5>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="card text-center shadow-sm">
                                    <div class="card-body p-2">
                                        <small class="text-muted">Clicks</small>
                                        <h5><?= $clicks ?></h5>
                                    </div>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="card text-center shadow-sm">
                                    <div class="card-body p-2">
                                        <small class="text-muted">Unsubs</small>
                                        <h5><?= $unsubs ?></h5>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <h6>Tracking Links (use in your email template)</h6>
                        <div class="mb-2">
                            <label class="form-label">OPEN</label>
                            <input class="form-control" value="<?= htmlspecialchars($open_link) ?>">
                        </div>
                        <div class="mb-2">
                            <label class="form-label">CLICK</label>
                            <input class="form-control" value="<?= htmlspecialchars($click_link) ?>">
                        </div>
                        <div class="mb-2">
                            <label class="form-label">UNSUBSCRIBE</label>
                            <input class="form-control" value="<?= htmlspecialchars($unsub_link) ?>">
                        </div>

                        <small class="text-muted">
                            Email variable is fixed: <code>%EMAIL%</code> — your ESP will replace it.
                        </small>

                    <?php else: ?>
                        <p class="text-muted">Select a campaign from the left to see its details and links.</p>
                    <?php endif; ?>
                </div>
            </div>

            <div class="card shadow-sm">
                <div class="card-header">Events for this Campaign (<?= count($detail_events) ?>)</div>
                <div class="card-body p-0" style="max-height:260px;overflow:auto;">
                    <table class="table table-sm table-striped mb-0">
                        <thead class="table-light">
                        <tr>
                            <th>Email</th>
                            <th>Country</th>
                            <th>Vertical</th>
                            <th>Action</th>
                            <th>Offer</th>
                            <th>Date</th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($detail_events as $r): ?>
                            <tr>
                                <td><?= htmlspecialchars($r[0] ?? '') ?></td>
                                <td><?= htmlspecialchars($r[1] ?? '') ?></td>
                                <td><?= htmlspecialchars($r[2] ?? '') ?></td>
                                <td><?= htmlspecialchars($r[5] ?? '') ?></td>
                                <td><?= htmlspecialchars($r[6] ?? '') ?></td>
                                <td><?= htmlspecialchars($r[8] ?? '') ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <?php if(!count($detail_events)): ?>
                            <tr><td colspan="6" class="text-center text-muted">No events yet.</td></tr>
                        <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        </div>
    </div>
</div>

<!-- New Campaign Modal -->
<div class="modal fade" id="newCampaignModal" tabindex="-1">
  <div class="modal-dialog">
    <form method="POST" class="modal-content">
      <input type="hidden" name="action" value="create_campaign">
      <div class="modal-header">
        <h5 class="modal-title">Create New Campaign</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div class="modal-body">
        <div class="mb-2">
          <label class="form-label">Campaign Name</label>
          <input type="text" name="name" class="form-control" required placeholder="My Amazing Campaign">
        </div>
        <div class="mb-2">
          <label class="form-label">Vertical</label>
          <select name="vertical" class="form-select">
            <option value="">(Optional)</option>
            <?php foreach($verticalOptions as $v): ?>
            <option value="<?= htmlspecialchars($v) ?>"><?= htmlspecialchars($v) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="mb-2">
          <label class="form-label">Offer URL</label>
          <input type="url" name="offer" class="form-control" required placeholder="https://offer.com">
        </div>
      </div>
      <div class="modal-footer">
        <button class="btn btn-primary">Create</button>
      </div>
    </form>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
